<?php
class MsManager {

    static function getPowerState( $id ) {
        $msInfoArray = ConfManager::getMsInfo( $id ) ;
        $ret_array = CommonUtility::stepRunProgramAtMs( array(
            "stepname"  => "GetPowerState",
            "program"   => "/usr/bin/vmware-cmd",
            "arguments" => array(
                "--server",   $msInfoArray[ "esxi_ip" ],
                "--username", $msInfoArray[ "esxi_username" ],
                "--password", $msInfoArray[ "esxi_password" ],
                $msInfoArray[ "image_path" ],
                "getstate"
        ) ) ) ;
        
        return join( "\n", $ret_array[ 1 ] ) ;
    }
    
    static function getVmwareToolsState( $id ) {
        $msInfoArray = ConfManager::getMsInfo( $id ) ;
        $ret_array = CommonUtility::stepRunProgramAtMs( array(
            "stepname"  => "QueryVmwareToolsStatus",
            "program"   => "/usr/bin/vmware-cmd",
            "arguments" => array(
                "--server",   $msInfoArray[ "esxi_ip" ],
                "--username", $msInfoArray[ "esxi_username" ],
                "--password", $msInfoArray[ "esxi_password" ],
                $msInfoArray[ "image_path" ],
                "gettoolslastactive"
        ) ) ) ;
        
        return join( "\n", $ret_array[ 1 ] ) ;
    }
    
    static function waitUntilPowerState( $id, $criteria, $argv = null ) {
        $retryCount    = 0 ;
        $retryCountMax = isset( $argv[ "retry_max" ] )      ? $argv[ "retry_max" ]      : 30 ;
        $retryInterval = isset( $argv[ "retry_interval" ] ) ? $argv[ "retry_interval" ] : 30 ;
        
        while( true ) {
            $powerState = self::getPowerState( $id ) ;
            CommonUtility::debug_print( "id='$id', powerState='$powerState'" ) ;
            
            if( strcasecmp( $powerState, "getstate() = $criteria" ) === 0 ) {
                break ;
            }
            
            CommonUtility::debug_print( "Sleep $retryInterval seconds..." ) ;
            sleep( $retryInterval ) ;
            
            $retryCount++ ;
            if( $retryCount > $retryCountMax ) {
                throw new Exception( "WaitUntilPowerStateTimeout" ) ;
            }
            
            CommonUtility::debug_print( "retry($retryCount/$retryCountMax)..." ) ;
        }
    }
    
    static function waitUntilVmwareToolsState( $id, $criteria, $argv = null ) {
        $retryCount    = 0 ;
        $retryCountMax = isset( $argv[ "retry_max" ] )      ? $argv[ "retry_max" ]      : 30 ;
        $retryInterval = isset( $argv[ "retry_interval" ] ) ? $argv[ "retry_interval" ] : 30 ;
        
        $statusMeaningArray = array(
            "gettoolslastactive() = 0"   => "VMware Tools is not installed or not running.",
            "gettoolslastactive() = 1"   => "Guest operating system is responding normally.",
            "gettoolslastactive() = 5"   => "Intermittent heartbeat. There might be a problem with the guest operating system.",
            "gettoolslastactive() = 100" => "No heartbeat. Guest operating system might have stopped responding."
        ) ;
        
        while( true ) {
            $vmwareToolsState = self::getVmwareToolsState( $id ) ;
            $statuMeaning = isset( $statusMeaningArray[ $vmwareToolsState ] ) ?
                                   $statusMeaningArray[ $vmwareToolsState ]   :
                                   "unknown" ;
            CommonUtility::debug_print( "id='$id', vmwareToolsState='$vmwareToolsState', meaning='$statuMeaning'" ) ;
            
            if( strcasecmp( $vmwareToolsState, "gettoolslastactive() = $criteria" ) === 0 ) {
                break ;
            }
            
            CommonUtility::debug_print( "Sleep $retryInterval seconds..." ) ;
            sleep( $retryInterval ) ;
            
            $retryCount++ ;
            if( $retryCount > $retryCountMax ) {
                throw new Exception( "WaitUntilVmwareToolsTimeout" ) ;
            }
            
            CommonUtility::debug_print( "retry($retryCount/$retryCountMax)..." ) ;
        }
    }
    
    static function waitUntilSshable( $id, $criteria = true, $argv = null ) {
        $retryCount    = 0 ;
        $retryCountMax = isset( $argv[ "retry_max" ] )      ? $argv[ "retry_max" ]      : 30 ;
        $retryInterval = isset( $argv[ "retry_interval" ] ) ? $argv[ "retry_interval" ] : 30 ;
        
        $msInfoArray = ConfManager::getMsInfo( $id ) ;
        
        while( true ) {
            $sshable = CommonUtility::isIpSshable( $msInfoArray[ "ip" ] ) ;
            CommonUtility::debug_print( "id='$id', ip='${msInfoArray[ "ip" ]}', sshable='$sshable'" ) ;
            
            if( $sshable == $criteria ) {
                break ;
            }
            
            CommonUtility::debug_print( "Sleep $retryInterval seconds..." ) ;
            sleep( $retryInterval ) ;
            
            $retryCount++ ;
            if( $retryCount > $retryCountMax ) {
                throw new Exception( "WaitUntilSshableTimeout" ) ;
            }
            
            CommonUtility::debug_print( "retry($retryCount/$retryCountMax)..." ) ;
        }
    }
    
    static function powerOnById( $id ) {
        if( $id == "1" ) {
            throw new Exception( "PowerOnNotSupport: ms_id='$id'" ) ;
        }
        
        $msInfoArray = ConfManager::getMsInfo( $id ) ;
        
        # get PowerState
        $powerState = self::getPowerState( $id ) ;
        CommonUtility::debug_print( "id='$id', powerState='$powerState'" ) ;
        
        # power-on when it is off
        if( strcasecmp( $powerState, "getstate() = off" ) == 0 ) {
            CommonUtility::stepRunProgramAtMs( array(
                "stepname"  => "PowerOn",
                "program"   => "/usr/bin/vmware-cmd",
                "arguments" => array(
                    "--server",   $msInfoArray[ "esxi_ip" ],
                    "--username", $msInfoArray[ "esxi_username" ],
                    "--password", $msInfoArray[ "esxi_password" ],
                    $msInfoArray[ "image_path" ],
                    "start"
            ) ) ) ;
        }
        
        # wait PowerState is on
        self::waitUntilPowerState( $id, "on" ) ;
        
        # wait VmwareTools is ready
        self::waitUntilVmwareToolsState( $id, "1" ) ;
    }
    
    static function powerOnByVmpath( $ms_vmpath ) {
        
    }
    
    static function powerOffById( $id ) {
        if( $id == "1" ) {
            throw new Exception( "PowerOffNotSupport: ms_id='$id'" ) ;
        }

        $msInfoArray = ConfManager::getMsInfo( $id ) ;
        
        # get PowerState
        $powerState = self::getPowerState( $id ) ;
        CommonUtility::debug_print( "id='$id', powerState='$powerState'" ) ;
        
        # power-off when it is on
        if( strcasecmp( $powerState, "getstate() = on" ) == 0 ) {
            CommonUtility::stepRunProgramAtMs( array(
                "stepname"  => "PowerOff",
                "program"   => "/usr/bin/vmware-cmd",
                "arguments" => array(
                    "--server",   $msInfoArray[ "esxi_ip" ],
                    "--username", $msInfoArray[ "esxi_username" ],
                    "--password", $msInfoArray[ "esxi_password" ],
                    $msInfoArray[ "image_path" ],
                    "stop",
                    "soft"
            ) ) ) ;
            
            CommonUtility::debug_print( "sleep 10 seconds" ) ;
            sleep( 10 ) ;
        }
        
        # wait PowerState is off
        self::waitUntilPowerState( $id, "off" ) ;
    }
    
    static function powerOffByVmpath( $ms_vmpath ) {
    }

    static function runProgramBySsh( $id, $argv ) {
        $msInfoArray = ConfManager::getMsInfo( $id ) ;

        $program    = $argv[ 'program' ] ;
        $arguments  = isset( $argv[ 'arguments' ] ) ? $argv[ 'arguments' ] : array() ;
        $isBlocking = isset( $argv[ 'is_block' ] )  ? $argv[ 'is_block' ]  : true ;
        $stepname   = isset( $argv[ 'stepname' ] )  ? $argv[ 'stepname' ]  : "runProgramBySsh" ;
        
        # escape arguments
        for( $i = 0 ; $i < count( $arguments ); $i++ ) {
            $arguments[ $i ] = escapeshellarg( $arguments[ $i ] ) ;
        }

        # compose command
        $command = join( " ", array( $program,
                                     join( " ", $arguments ),
                                     "; echo \$?" ) ) ;
        CommonUtility::debug_print( "command='$command'" ) ;

        # get ssh connection
        $connection = CommonUtility::getSshConnection(
                            $msInfoArray[ "ip" ],
                            $msInfoArray[ "username" ],
                            $msInfoArray[ "password" ] ) ;

        # execute
        $stdoutStream = ssh2_exec( $connection, $command ) ;
        $stderrStream = ssh2_fetch_stream( $stdoutStream, SSH2_STREAM_STDERR ) ;

        if( $stdoutStream === false ) {
            $msg = "ExecuteCommandFail: command='$command'" ;
            CommonUtility::debug_print( $msg ) ;
            return array( "1", array( $msg ) ) ;
        }

        # wait for execution result
        $retValue = "0" ;
        $outMsgArray = array() ;
        $errMsgArray = array() ;
        if( $isBlocking === true ) {
            CommonUtility::debug_print( "WaitingForSshResult..." ) ;
            stream_set_blocking( $stdoutStream, 1 ) ;
            
            $outMsgArray = preg_split( "/\n+/", trim( stream_get_contents( $stdoutStream ) ) ) ;
            $errMsgArray = preg_split( "/\n+/", trim( stream_get_contents( $stderrStream ) ) ) ;
            CommonUtility::debug_print( "ReceivedSshResult..." ) ;
            
            $retValue = array_pop( $outMsgArray ) ;
        }
        fclose( $stdoutStream ) ;
        fclose( $stderrStream ) ;

        return array( $retValue,
                      $outMsgArray,
                      $errMsgArray ) ;
    }

    static function runProgramByVmware( $id, $argv ) {
        $msInfoArray = ConfManager::getMsInfo( $id ) ;
        
        $program    = $argv[ 'program' ] ;
        $arguments  = isset( $argv[ 'arguments' ] ) ? $argv[ 'arguments' ] : array() ;
        $outputPath = isset( $argv[ 'pipe_to' ] )   ? $argv[ 'pipe_to' ]   : null ;
        $isEscape   = isset( $argv[ 'escape' ] )    ? $argv[ 'escape' ]    : 1 ;
        $stepname   = isset( $argv[ 'stepname' ] )  ? $argv[ 'stepname' ]  : "runProgramByVmware" ;
        
        $newArgumentsArray = array(
            "-T",  "server",
            "-h",  $msInfoArray[ "esxi_ip" ],
            "-u",  $msInfoArray[ "esxi_username" ],
            "-p",  $msInfoArray[ "esxi_password" ],
            "-gu", $msInfoArray[ "username" ],
            "-gp", $msInfoArray[ "password" ],
            "runProgramInGuest",
            $msInfoArray[ "image_path_vix" ],
            $program
        ) ;
        
        foreach( $arguments as $argument ) {
            array_push( $newArgumentsArray, $argument ) ;
        }
        
        CommonUtility::stepRunProgramAtMs( array(
            "stepname"  => $stepname,
            "pipe_to"   => $outputPath,
            "escape"    => $isEscape,
            
            "program"   => "/usr/bin/vmrun",
            "arguments" => $newArgumentsArray
        ),
            array('-u', '-p', '-gu', '-gp')
        ) ;
    }
    
    static function runProgram( $id, $argv ) {
        $msInfoArray = ConfManager::getMsInfo( $id ) ;
        
        if( CommonUtility::isIpSshable( $msInfoArray[ "ip" ] ) == true ) {
            return self::runProgramBySsh( $id, $argv ) ;
        }
        else {
            return self::runProgramByVmware( $id, $argv ) ;
        }
    }
    
    static function stepRunProgramBySsh( $id, $argv ) {
        $stepName = isset( $argv[ "stepname" ] ) ? $argv[ "stepname" ] : "RunProgramAtMs" ;
        
        # composing debug message
        $msg = $stepName . ": " . escapeshellarg( $argv[ 'program' ] ) ;
        $argumentsArray = isset( $argv[ 'arguments' ] ) ? $argv[ 'arguments' ] : array() ;
        foreach( $argumentsArray as $argument ) {
            $msg .=  " " . escapeshellarg( $argument ) ;
        }
        CommonUtility::debug_print( $msg ) ;
        
        # launching
        $ret_array = self::runProgramBySsh( $id, $argv ) ;
        
        # verify return code
        $retCode      = $ret_array[ 0 ] ;
        $stdoutMsgAry = $ret_array[ 1 ] ;
        $stdoutMsg    = join( "\n", $stdoutMsgAry ) ;
        if( $retCode != 0 ) {
            $stderrMsgAry = $ret_array[ 2 ] ;
            $stderrMsg    = join( "\n", $stderrMsgAry ) ;
            throw new Exception( "${stepName}Fail: ret=$retCode, msg='$stderrMsg'" ) ;
        }
        CommonUtility::debug_print( "${stepName}Pass: msg='$stdoutMsg'" ) ;
        
        return $ret_array ;
    }
    
    static function setNetworkConfiguration( $id, $argv ) {
        $msInfoArray = ConfManager::getMsInfo( $id ) ;
        
        $nic     = isset( $argv[ "nic" ] )     ? $argv[ "nic" ]     : "eth0" ;
        $dhcp    = isset( $argv[ "dhcp" ] )    ? $argv[ "dhcp" ]    : "1" ;
        $ip      = isset( $argv[ "ip" ] )      ? $argv[ "ip" ]      : null ;
        $mask    = isset( $argv[ "mask" ] )    ? $argv[ "mask" ]    : null ;
        $gateway = isset( $argv[ "gateway" ] ) ? $argv[ "gateway" ] : null ;
        $dns     = isset( $argv[ "dns" ] )     ? $argv[ "dns" ]     : null ;
        
        $ipConf = "" ;
        if( $dhcp == "1" ) {
            $ipConf = "DEVICE=$nic\n" .
                      "BOOTPROTO=dhcp\n" . 
                      "ONBOOT=yes\n" ;
        }
        else {
            $ipConf = "DEVICE=$nic\n" .
                      "BOOTPROTO=static\n" . 
                      "ONBOOT=yes\n" .
                      "IPADDR=$ip\n" .
                      "NETMASK=$mask\n" .
                      "GATEWAY=$gateway\n" ;
        }

        self::runProgramByVmware( $id, array(
            "stepname"  => "ModifyMsNetwork",
            "pipe_to"   => "/etc/sysconfig/network-scripts/ifcfg-$nic",
            "program"   => "/bin/echo",
            "arguments" => array( $ipConf )
        ) ) ;
        
        self::runProgramByVmware( $id, array(
            "stepname"  => "StopMsNetworkAdapter1",
            "program"   => "/sbin/ifdown",
            "arguments" => array( $nic )
        ) ) ;
        
        self::runProgramByVmware( $id, array(
            "stepname"  => "StartMsNetworkAdapter1",
            "program"   => "/sbin/ifup",
            "arguments" => array( $nic )
        ) ) ;
        
        # self::waitUntilSshable( $id ) ;
        
        self::runProgramByVmware( $id, array(
            "stepname"  => "UpdateDNS",
            "pipe_to"   => "/etc/resolv.conf",
            "program"   => "/bin/echo",
            "arguments" => array( "nameserver $dns\n" )
        ) ) ;
    }
}

if( realpath( $_SERVER[ "SCRIPT_NAME" ] ) == realpath( __FILE__ ) ) {
    require_once( "Constants.class.php" ) ;
    require_once( "CommonUtility.class.php" ) ;
    require_once( "ConfManager.class.php" ) ;
    require_once( "DbManager.class.php" ) ;

    try {
        // ScManager::powerOffById( "1" ) ;
        // ScManager::powerOnById( "1" ) ;
        // ScManager::runProgramByVmware( "1", array(
            // "program" => "/bin/touch",
            // "arguments" => array(
                // "/tmp/test1",
                // "/tmp/test2"
        // ) ) ) ;
        MsManager::setNetworkConfiguration( "1", array(
            "nic"     => "eth0",
            "dhcp"    => "0",
            "ip"      => "169.254.3.3",
            "mask"    => "255.255.255.0",
            "gateway" => "169.254.3.2",
            "dns"     => "169.254.3.2"
        ) ) ;
    }
    
    catch( Exception $e ) {
        echo "[" . $e -> getFile() . "(" . $e -> getLine() . ")]: " . $e -> getMessage() . "\n" ;
    }
}
